#!/usr/bin/env python3

# Author: 	Daniel Bosk <daniel.bosk@miun.se>
# Date: 	17 November 2013

# Revised by: 	Jimmy Åhlander
# Date:		Winter 2015
# Comment:	The algorithm was rewritten from scratch as it did not conform 
#		to any known model of ESC. It is now based on a single loop 
#		with a worst-case of three iterations. It does not mimic SC 
#               even if only read operations are performed. E.g. 1 2 3 4 2 5 6.

# Copyright (c) 2013, Daniel Bosk <daniel.bosk@miun.se>
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met: 
#
#  - Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
#
#  - Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
#
#  - Neither the name of the University nor the names of its contributors
#    may be used to endorse or promote products derived from this software
#    without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
# LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
# CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
# SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
# INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
# CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
# ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.

import sys

queue = []
qhead = 0

def remove_from_queue( n ):
	if ( n >= len( queue ) ):
		raise Exception( "cannot remove element outside of queue" )
	i = ( qhead + n ) % len(queue)
	while ( i != qhead ):
		queue[i] = queue[ (i+1) % len(queue) ]
		i = ( i + 1 ) % len(queue)

def memalg_esc( osmem, p, pt ):
	global qhead

	if ( pt[p].valid ):
		return

	sys.stdout.write( "page " + str( p ) + " generated page fault\n" )

	# if we still have free frames, use one of these
	if ( osmem.free_total() > 0 ):
		pt[p].frame = osmem.get_free_frame()
		sys.stdout.write( "page " + str( p ) + " allocated to free frame " + str( pt[p].frame ) + "\n" )

		queue.append( p )
	# otherwise swap out one page
	else:
		# we start at qhead, so it's best so far
		best_idx = qhead

		# first loop, search for class (0,0)
		bestPageFound = False
		if ( not bestPageFound ):
			current = qhead
			for i in range( len( queue ) ):
				page = pt[ queue[ current ] ]

				if ( not page.referenced and not page.modified ):
					best_idx = current
					bestPageFound = True
					break;

				current = ( current + 1 ) % len( queue )

		# second loop, search for class (0,1), reset ref bits encountered
		if ( not bestPageFound ):
			for i in range( len( queue ) ):
				cur = qhead
				page = pt[ queue[ cur ] ]

				if ( not page.referenced and page.modified):
					best_idx = cur
					bestPageFound = True
					break;

				# reset the referenced bit
				page.referenced = False
				qhead = ( qhead + 1 ) % len( queue )

		# third loop, search for class (0,0) again
		if ( not bestPageFound ):
			for i in range( len( queue ) ):
				cur = qhead
				page = pt[ queue[ cur ] ]

				if ( not page.referenced and not page.modified ):
					best_idx = cur
					bestPageFound = True
					break;

				qhead = ( qhead + 1 ) % len( queue )

		# if the third loop finishes without finding any specific page
		# it means all pages were of class (1,1). We can therefore
		# safely choose qhead as victim.
		qhead = best_idx

		# swap-out the best choice
		pn = queue[ qhead ]
		pt[pn].valid = False
		osmem.swap_out( pn )
		# allocate newly freed frame
		pt[p].frame = pt[pn].frame

		# update the queue, add the new page to the end of the queue
		queue[ qhead ] = p
		qhead = ( qhead + 1 ) % len( queue )

	# actually swap in the page
	osmem.swap_in( p )
	pt[p].valid = True
	pt[p].modified = False
	pt[p].referenced = False

